function Mt=process_ftA(kvAb,kvBa,KAb,KBa,NA0,NB0,epA,epB,L,V,Cas,presnost,gr)
%Mt=process_ftA(kvAb,kvBa,KAb,KBa,NA0,NB0,epA,epB,L,V,Cas,gr)
%Calculates the evolution of concentration of substances A and B that can be transformed (A <-> B)
% by photochemical and thermal reactions. 
%"kvAb" is the quantum yield of the photoreaction A -> B (probability of reaction after photon absorption by the substance A)
%"kvBa" is the quantum yield of the photoreaction B -> A 
%"KAb" is the rate constant of the thermal transformation A -> B
%"KBa" is the rate constant of the thermal transformation B -> A
%"NA0" is the initial concentration of the substance A 
%"NB0" is the initial concentration of the substance B 
%"epA" is the absorption cross section for absortpion of a photon by the substance A [m2] 
%"epB" is the absorption cross section for absortpion of a photon by the substance B [m2]
%"L" is the length of the cuvette (i.e. the length of the light path in the sample) [m], 
%"V" is the sample volume [m3]
%"Cas" is a matrix with two columns: 
%   1. column contains times of measurements, 
%   2. column are numbers of photons impacted on the sample. 
%"presnost" is a parameter of the numerical calculation that defines the calculation step. 
%   Higher value of "presnost" leads to a more precise calculation. 
%   (To a shorter calculation step - to a shorter step in number of impacted photons). 
%If gr=1, the result will be shown in a figure. 
%"Mt" is a resulting matrix with four columns: 
%   1. column is time [s], 
%   2. column is number of impacted photons,
%   3. column is the concentration of the substance A [m-3], 
%   4. column is the concentration of the substance B [m-3] 



N0=NA0+NB0; %Total concentration of both substances A and B 

%Determination of the integration step: 
dN=(1-exp(-0.5*N0*epA*L-0.5*N0*epB*L)) * max([kvBa*epB kvAb*epA]) / (epA+epB);
dN=dN/V; 
dq=N0/dN*10^(-presnost); %the calculation step

Mt=[0 0 NA0 NB0];
Qmax=Cas(end,2); 
n=ceil(Qmax/dq);  %number of steps
Dt=dq*(Cas(2:end,1)-Cas(1:end-1,1))./(Cas(2:end,2)-Cas(1:end-1,2)); 
t=0; 

%The integration: 
for j=1:n, 
 %Concentration changes caused by photochemical reactions:
 dN=dq* (1-exp(-Mt(j,3)*epA*L-Mt(j,4)*epB*L)) * (kvBa*Mt(j,4)*epB - kvAb*Mt(j,3)*epA) / (Mt(j,3)*epA+Mt(j,4)*epB);
 dN=dN/V; 
 %Addition of concentration changes caused by thermal reactions:
 qact=dq*j; 
 prirazenit=round(sum(sign(qact-Cas(:,2))+1)/2); %Determines, in which interval between measurements is the actual point, i.e. what time step will be used.
 if prirazenit>length(Dt)
  prirazenit=prirazenit-1; 
 end
 dt=Dt(prirazenit); 
 dN=dN + KBa*Mt(j,4)*dt - KAb*Mt(j,3)*dt; 
 %The total change: 
 N=Mt(j,3)+dN;
 %Data are placed into the matrix "Mt": 
 t=t+dt; 
 Mt(j+1,:)=[t qact N N0-N];
end

if gr==1
 %The limit: 
 limpom=kvBa*epB/(kvAb*epA); %Ratio of A and B concentrations in the photostationary state (if there were no thermal reactions). 
 limA=limpom*N0/(1+limpom); %Concentration of the substance A in the photostationary state
 limB=N0/(1+limpom); %Concentration of the substance B in the photostationary state

 %Graph of the dependence on the number of impacted photons: 
 figure; plot(Mt(:,2),Mt(:,3),'b',Mt(:,2),Mt(:,4),'r',Qmax,limA,'bo',Qmax,limB,'ro'); grid on;
 legend('A','B'); xlabel('Number of impacted photons'); ylabel('Concentration [m^{-3}]'); 
 
 %Graph of the dependence on time: 
 figure; plot(Mt(:,1),Mt(:,3),'b',Mt(:,1),Mt(:,4),'r'); grid on;
 legend('A','B'); xlabel('Time [s]'); ylabel('Concentration [m^{-3}]'); 
end
